<?php
/**
 * Plugin Name: Blinkz Q&A
 * Description: Blinkz plugin for showing Q&A knowledge articles
 * Version: 1.0.2
 * Author: Blijnder.nl
 * Author URI: https://blijnder.nl
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Blinkz Plugin Class
 */
class BlinkzPlugin
{
    private static $instance = null;
    
    // Default intro text
    const DEFAULT_INTRO_TEXT = 'Hieronder vind je antwoorden op veelgestelde vragen. Klik op een categorie om de vragen uit die categorie te zien, of bekijk alle vragen door op <i>Alles</i> te klikken.';

    // Default API URL
    const DEFAULT_API_URL = 'https://blinkz-api.blijnder.nl/api/knowledge';

    // Default settings for newest/most-read articles
    const DEFAULT_SHOW_NEWEST = true;
    const DEFAULT_SHOW_MOST_READ = false;
    const DEFAULT_ARTICLES_AMOUNT = 5;

    // Default CSS constant
    const DEFAULT_CSS = '.blinkz-item-title {
    cursor: pointer;
    margin-bottom: 10px;
    user-select: none;
    display: flex;
    justify-content: space-between;
    align-items: center;
}
.blinkz-item-title:hover {
    opacity: 0.7;
}
.blinkz-toggle {
    font-size: 12px;
    transition: transform 0.3s ease;
}
.blinkz-toggle.open {
    transform: rotate(180deg);
}
.blinkz-item-content {
    max-height: 0;
    overflow: hidden;
    transition: max-height 0.3s ease-in-out;
    margin-bottom: 15px;
}
.blinkz-item-content.open {
    max-height: 5000px;
}
.blinkz-question, .blinkz-answer {
    margin-bottom: 10px;
    padding: 15px;
    background: #f9f9f9;
    border-left: 3px solid #0073aa;
    font-size: var(--wp--preset--font-size--medium);
    line-height: 1.6;
    overflow: visible;
}
.blinkz-answer {
    position: relative;
}
.blinkz-answer img {
    max-width: 100%;
    height: auto;
    margin: 10px 0;
    border-radius: 4px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    display: block;
}
.blinkz-answer p {
    margin-bottom: 15px;
}
.blinkz-answer p:last-child {
    margin-bottom: 0;
}
.blinkz-meta-info {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-top: 15px;
}
.blinkz-item-categories {
    font-size: var(--wp--preset--font-size--small);
    flex: 1;
}
.blinkz-category-link.active {
    font-weight: bold;
}
.blinkz-intro-section {
    margin-bottom: 20px;
    font-size: var(--wp--preset--font-size--medium);
    line-height: 1.6;
}
.blinkz-intro-section p {
    margin-bottom: 10px;
    font-size: inherit;
    line-height: inherit;
}
.blinkz-intro-section p:last-child {
    margin-bottom: 0;
}
.blinkz-attribution {
    flex-shrink: 0;
}
.blinkz-logo {
    max-width: 100px;
    max-height: 30px;
    width: auto;
    height: auto;
    opacity: 1;
    display: block;
}
.blinkz-featured-section {
    margin-bottom: 20px;
    text-align: left;
    font-size: var(--wp--preset--font-size--medium);
    line-height: 1.6;
}
.blinkz-featured-title {
    margin: 0 0 8px 0;
    font-size: 0.9em;
    font-weight: 600;
    color: inherit;
}
.blinkz-featured-list {
    list-style: none;
    margin: 0;
    padding: 0;
    text-align: left;
}
.blinkz-featured-item {
    text-align: left;
}
.blinkz-featured-link {
    color: inherit !important;
    text-decoration: none !important;
    font-size: var(--wp--preset--font-size--medium) !important;
    line-height: 1.6 !important;
    font-weight: 300 !important;
    cursor: pointer;
}
.blinkz-featured-link:hover {
    text-decoration: underline !important;
}
.blinkz-featured-link:visited {
    color: inherit !important;
}';

    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        add_action('admin_menu', [$this, 'addAdminMenu']);
        add_action('admin_init', [$this, 'initSettings']);
        add_action('admin_post_blinkz_save_config', [$this, 'saveConfiguration']);
        add_action('init', [$this, 'registerBlock']);
        add_action('enqueue_block_editor_assets', [$this, 'enqueueBlockAssets']);
        add_action('wp_ajax_blinkz_restore_default_css', [$this, 'restoreDefaultCss']);
        add_action('wp_ajax_blinkz_restore_default_intro', [$this, 'restoreDefaultIntro']);
        add_action('wp_ajax_blinkz_restore_default_api_url', [$this, 'restoreDefaultApiUrl']);
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), [$this, 'addSettingsLink']);
    }

    /**
     * Register the Blinkz block
     */
    public function registerBlock()
    {
        register_block_type('blinkz/blinkz-knowledge', [
            'editor_script' => 'blinkz-block-editor',
            'render_callback' => [$this, 'renderBlock'],
        ]);
    }

    /**
     * Enqueue block editor assets
     */
    public function enqueueBlockAssets()
    {
        wp_enqueue_script(
            'blinkz-block-editor',
            plugin_dir_url(__FILE__) . 'block.js',
            ['wp-blocks', 'wp-element', 'wp-editor'],
            '1.0.0',
            true
        );
    }

    /**
     * Get data from the API endpoint
     */
    public function getApiData()
    {
        $options = get_option('blinkz_options', []);
        $api_url = Arr::get($options, 'api_url', self::DEFAULT_API_URL);

        // Check if API URL is configured
        if (empty($api_url)) {
            return [
                'success' => false,
                'error' => 'API URL not configured',
                'data' => null
            ];
        }

        // Set up request arguments
        $args = [
            'timeout' => 30,
            'headers' => [
                'Accept' => 'application/json',
                'User-Agent' => 'Blinkz WordPress Plugin/1.0.2'
            ]
        ];

        // Make the HTTP request
        $response = wp_remote_get($api_url, $args);

        // Check for WP_Error
        if (is_wp_error($response)) {
            return [
                'success' => false,
                'error' => 'Request failed: ' . $response->get_error_message(),
                'data' => null
            ];
        }

        // Get response code
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        // Check if request was successful
        if ($response_code !== 200) {
            return [
                'success' => false,
                'error' => 'HTTP Error: ' . $response_code,
                'data' => null
            ];
        }

        // Try to decode JSON response
        $data = json_decode($response_body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return [
                'success' => false,
                'error' => 'Invalid JSON response: ' . json_last_error_msg(),
                'data' => $response_body // Return raw response for debugging
            ];
        }

        return [
            'success' => true,
            'error' => null,
            'data' => $data
        ];
    }

    /**
     * Clean empty spans and empty containers from content
     */
    private function cleanEmptySpans($content)
    {
        return $content;
        // Use DOMDocument to parse and traverse all elements
        $dom = new DOMDocument();
        // Suppress warnings due to malformed HTML
        @$dom->loadHTML('<?xml encoding="utf-8" ?>' . $content, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

        foreach (iterator_to_array($dom->getElementsByTagName('*')) as $element) {
            // Remove <span> that is empty or only whitespace/nbsp
            if (
                $element->tagName === 'span' &&
                trim(str_replace("\xC2\xA0", '', $element->textContent)) === ''
            ) {
                $element->parentNode->removeChild($element);
            }
            // Remove <p> or <div> that is empty or only whitespace/nbsp (after span removal)
            if (
                in_array($element->tagName, ['p', 'div']) &&
                trim(str_replace("\xC2\xA0", '', $element->textContent)) === ''
            ) {
                $element->parentNode->removeChild($element);
            }
        }

        $content = $dom->saveHTML();
        
        return $content;
    }

    /**
     * Render the block on frontend
     */
    public function renderBlock($attributes, $content)
    {
        $api_result = $this->getApiData();
        
        // Handle API errors
        if (!$api_result['success']) {
            return '<div class="blinkz-knowledge-block"><p>Error loading knowledge data: ' . esc_html($api_result['error']) . '</p></div>';
        }
        
        $knowledge_data = Arr::get($api_result['data'], 'knowledge', []);
        
        // Extract unique categories
        $categories = [];
        foreach ($knowledge_data as $item) {
            if (Arr::has($item, 'categories') && is_array(Arr::get($item, 'categories'))) {
                foreach (Arr::get($item, 'categories') as $category) {
                    if (Arr::has($category, 'slug') && Arr::has($category, 'name')) {
                        $slug = Arr::get($category, 'slug');
                        $name = Arr::get($category, 'name');
                        if (!Arr::has($categories, $slug)) {
                            $categories[$slug] = $name;
                        }
                    }
                }
            }
        }
        
        // Generate unique ID for this block instance
        $block_id = 'blinkz-knowledge-' . uniqid();
        
        // Build the HTML
        $html = '<div class="blinkz-knowledge-block" id="' . esc_attr($block_id) . '">';
        
        // Get options
        $options = get_option('blinkz_options', []);
        $intro_text = Arr::get($options, 'intro_text', self::DEFAULT_INTRO_TEXT);
        $show_newest = Arr::get($options, 'show_newest', self::DEFAULT_SHOW_NEWEST);
        $show_most_read = Arr::get($options, 'show_most_read', self::DEFAULT_SHOW_MOST_READ);
        $articles_amount = (int) Arr::get($options, 'articles_amount', self::DEFAULT_ARTICLES_AMOUNT);

        // Display intro text with standard suffix
        if (!empty($intro_text) || !empty($categories)) {
            $html .= '<div class="blinkz-intro-section">';

            // Custom intro text
            if (!empty($intro_text)) {
                $html .= '<p>' . wp_kses_post($intro_text) . '</p>';
            }

            // Standard suffix with link
            $html .= '<p>Staat jouw vraag er niet bij, neem dan contact op met het <a href="https://helpdeskdigitalezorg.nl/blinkz/" target="_blank" rel="noopener noreferrer">Blinkz Informatiepunt</a>.</p>';
            $html .= '</div>';
        }

        // Newest articles section (sorted by created_at timestamp, newest first)
        if ($show_newest && !empty($knowledge_data) && $articles_amount > 0) {
            $sorted_by_date = $knowledge_data;
            usort($sorted_by_date, function ($a, $b) {
                $date_a = Arr::get($a, 'created_at', 0);
                $date_b = Arr::get($b, 'created_at', 0);
                return $date_b <=> $date_a; // Descending (newest first)
            });
            $newest_articles = array_slice($sorted_by_date, 0, $articles_amount);
            $html .= '<div class="blinkz-featured-section blinkz-newest-section">';
            $html .= '<h3 class="blinkz-featured-title">Nieuwste artikelen</h3>';
            $html .= '<ul class="blinkz-featured-list">';
            foreach ($newest_articles as $article) {
                $title = Arr::get($article, 'title', '');
                if (!empty($title)) {
                    $html .= '<li class="blinkz-featured-item" data-article-title="' . esc_attr($title) . '">';
                    $html .= '<a href="#" class="blinkz-featured-link">' . esc_html($title) . '</a>';
                    $html .= '</li>';
                }
            }
            $html .= '</ul>';
            $html .= '</div>';
        }

        // Most read articles section (sorted by view_count, highest first)
        if ($show_most_read && !empty($knowledge_data) && $articles_amount > 0) {
            $sorted_by_views = $knowledge_data;
            usort($sorted_by_views, function ($a, $b) {
                $views_a = (int) Arr::get($a, 'view_count', 0);
                $views_b = (int) Arr::get($b, 'view_count', 0);
                return $views_b <=> $views_a; // Descending (most views first)
            });
            $most_read_articles = array_slice($sorted_by_views, 0, $articles_amount);
            $html .= '<div class="blinkz-featured-section blinkz-most-read-section">';
            $html .= '<h3 class="blinkz-featured-title">Meest gelezen</h3>';
            $html .= '<ul class="blinkz-featured-list">';
            foreach ($most_read_articles as $article) {
                $title = Arr::get($article, 'title', '');
                if (!empty($title)) {
                    $html .= '<li class="blinkz-featured-item" data-article-title="' . esc_attr($title) . '">';
                    $html .= '<a href="#" class="blinkz-featured-link">' . esc_html($title) . '</a>';
                    $html .= '</li>';
                }
            }
            $html .= '</ul>';
            $html .= '</div>';
        }

        // Categories menu at the top
        if (!empty($categories)) {
            $html .= '<div class="blinkz-categories-menu">';
            $html .= '<small>';
            
            // Add "All" link to show all articles (no active class by default)
            $html .= '<a href="#all" data-category="all" class="blinkz-category-link">Alles</a> • ';
            
            $category_links = [];
            foreach ($categories as $slug => $name) {
                $display_name = str_replace('_', ' ', $name);
                $category_links[] = '<a href="#' . esc_attr($slug) . '" data-category="' . esc_attr($slug) . '" class="blinkz-category-link">' . esc_html($display_name) . '</a>';
            }
            
            $html .= implode(' • ', $category_links);
            $html .= '</small>';
            $html .= '</div>';
            $html .= '<hr class="blinkz-separator" style="display: none;">';
        }
        
        // Knowledge articles list
        $html .= '<div class="blinkz-knowledge-list">';
        
        if (!empty($knowledge_data)) {
            foreach ($knowledge_data as $index => $item) {
                $title = Arr::get($item, 'title', '');
                $question = Arr::get($item, 'bc_question__c', '');
                $answer = Arr::get($item, 'bc_answer__c', '');
                $item_categories = Arr::get($item, 'categories', []);
                
                // Create category slugs for filtering
                $category_slugs = [];
                foreach ($item_categories as $category) {
                    if (Arr::has($category, 'slug')) {
                        $category_slugs[] = Arr::get($category, 'slug');
                    }
                }
                
                $item_id = 'blinkz-item-' . $index;
                
                // Items are hidden by default, shown only when a category is selected
                $html .= '<div class="blinkz-knowledge-item" data-categories="' . esc_attr(implode(',', $category_slugs)) . '" style="display: none;">';
                
                // Title (clickable)
                if (!empty($title)) {
                    $html .= '<h4 class="blinkz-item-title" data-target="' . esc_attr($item_id) . '">' . esc_html($title) . ' <span class="blinkz-toggle">▼</span></h4>';
                }
                
                // Content (initially hidden)
                $html .= '<div class="blinkz-item-content" id="' . esc_attr($item_id) . '">';
                
                // Question (if different from title and exists)
                if (!empty($question) && strip_tags($question) !== $title) {
                    //$html .= '<div class="blinkz-question"><strong>Question:</strong> ' . wp_kses_post($question) . '</div>';
                }
                
                // Answer
                if (!empty($answer)) {
                    $html .= '<div class="blinkz-answer">' . wp_kses_post($this->cleanEmptySpans($answer));


                    // Categories and logo attribution
                    $html .= '<div class="blinkz-meta-info">';
                    
                    // Categories for this item
                    if (!empty($item_categories)) {
                        $html .= '<div class="blinkz-item-categories"><small>Categories: ';
                        $cat_names = [];
                        foreach ($item_categories as $category) {
                            if (Arr::has($category, 'name')) {
                                $cat_names[] = str_replace('_', ' ', Arr::get($category, 'name'));
                            }
                        }
                        $html .= esc_html(implode(', ', $cat_names));
                        $html .= '</small></div>';
                    }
                    
                    // Blinkz logo attribution
                    $logo_url = plugin_dir_url(__FILE__) . 'assets/img/blinkz_logo.png';
                    $html .= '<div class="blinkz-attribution">';
                    $html .= '<a href="https://blinkz.nu" target="_blank" rel="noopener noreferrer" title="Bron: Blinkz">';
                    $html .= '<img src="' . esc_url($logo_url) . '" alt="Blinkz" class="blinkz-logo" />';
                    $html .= '</a>';
                    $html .= '</div>';
                    
                    $html .= '</div>'; // End meta info

                    $html .= '</div>';


                }
                

                
                $html .= '</div>'; // End content
                $html .= '</div>'; // End item
            }
        } else {
            $html .= '<p>No knowledge articles found.</p>';
        }
        
        $html .= '</div>';
        $html .= '</div>';
        
        // Get custom CSS from options or use default
        $options = get_option('blinkz_options', []);
        $custom_css = Arr::get($options, 'custom_css', self::DEFAULT_CSS);
        
        // Add CSS for animations and styling
        $html .= '<style>' . $custom_css . '</style>';
        
        // Add JavaScript for filtering and expand/collapse
        $html .= '<script>
        (function() {
            const blockElement = document.getElementById("' . esc_js($block_id) . '");
            if (!blockElement) return;
            
            const categoryLinks = blockElement.querySelectorAll(".blinkz-category-link");
            const knowledgeItems = blockElement.querySelectorAll(".blinkz-knowledge-item");
            const itemTitles = blockElement.querySelectorAll(".blinkz-item-title");
            const separator = blockElement.querySelector(".blinkz-separator");

            // Category filtering function
            function filterItems(selectedCategory) {
                // Remove active class from all links
                categoryLinks.forEach(link => link.classList.remove("active"));

                // Add active class to selected link
                const activeLink = blockElement.querySelector(`[data-category="${selectedCategory}"]`);
                if (activeLink) {
                    activeLink.classList.add("active");
                }

                // Show separator when a category is selected
                if (separator) {
                    separator.style.display = "block";
                }

                // Filter items
                knowledgeItems.forEach(item => {
                    const itemCategories = item.getAttribute("data-categories");

                    if (selectedCategory === "all" || itemCategories.includes(selectedCategory)) {
                        item.style.display = "block";
                    } else {
                        item.style.display = "none";
                        // Close any open content when hiding items
                        const content = item.querySelector(".blinkz-item-content");
                        const toggle = item.querySelector(".blinkz-toggle");
                        if (content && content.classList.contains("open")) {
                            content.classList.remove("open");
                            if (toggle) toggle.classList.remove("open");
                        }
                    }
                });
            }
            
            // Add click event listeners to category links
            categoryLinks.forEach(link => {
                link.addEventListener("click", function(e) {
                    e.preventDefault();
                    const category = this.getAttribute("data-category");
                    filterItems(category);
                    
                    // Update URL hash
                    if (category === "all") {
                        history.pushState(null, null, "#");
                    } else {
                        history.pushState(null, null, "#" + category);
                    }
                });
            });
            
            // Add click event listeners to article titles
            itemTitles.forEach(title => {
                title.addEventListener("click", function() {
                    const targetId = this.getAttribute("data-target");
                    const content = document.getElementById(targetId);
                    const toggle = this.querySelector(".blinkz-toggle");
                    
                    if (content && toggle) {
                        const isCurrentlyOpen = content.classList.contains("open");
                        
                        // Close all articles first
                        itemTitles.forEach(otherTitle => {
                            const otherTargetId = otherTitle.getAttribute("data-target");
                            const otherContent = document.getElementById(otherTargetId);
                            const otherToggle = otherTitle.querySelector(".blinkz-toggle");
                            
                            if (otherContent && otherToggle) {
                                otherContent.classList.remove("open");
                                otherToggle.classList.remove("open");
                            }
                        });
                        
                        // If the clicked article was not open, open it
                        if (!isCurrentlyOpen) {
                            content.classList.add("open");
                            toggle.classList.add("open");
                        }
                    }
                });
            });
            
            // Check for hash on page load
            function checkHashOnLoad() {
                const hash = window.location.hash.substring(1);
                if (hash && hash !== "") {
                    const targetLink = blockElement.querySelector(`[data-category="${hash}"]`);
                    if (targetLink) {
                        filterItems(hash);
                    }
                }
            }

            // Function to open a specific article by title
            function openArticleByTitle(articleTitle) {
                // Show all items first (select "all" category)
                filterItems("all");

                // Find the article with matching title
                let foundItem = null;
                knowledgeItems.forEach(item => {
                    const title = item.querySelector(".blinkz-item-title");
                    if (title && title.textContent.trim().replace(/\\s*▼\\s*$/, "") === articleTitle) {
                        foundItem = item;
                    }
                });

                if (foundItem) {
                    // Close all articles first
                    itemTitles.forEach(otherTitle => {
                        const otherTargetId = otherTitle.getAttribute("data-target");
                        const otherContent = document.getElementById(otherTargetId);
                        const otherToggle = otherTitle.querySelector(".blinkz-toggle");
                        if (otherContent && otherToggle) {
                            otherContent.classList.remove("open");
                            otherToggle.classList.remove("open");
                        }
                    });

                    // Open the found article
                    const title = foundItem.querySelector(".blinkz-item-title");
                    const targetId = title.getAttribute("data-target");
                    const content = document.getElementById(targetId);
                    const toggle = title.querySelector(".blinkz-toggle");

                    if (content && toggle) {
                        content.classList.add("open");
                        toggle.classList.add("open");
                    }

                    // Scroll to the article
                    foundItem.scrollIntoView({ behavior: "smooth", block: "start" });
                }
            }

            // Add click event listeners to featured article links
            const featuredLinks = blockElement.querySelectorAll(".blinkz-featured-link");
            featuredLinks.forEach(link => {
                link.addEventListener("click", function(e) {
                    e.preventDefault();
                    const articleTitle = this.closest(".blinkz-featured-item").getAttribute("data-article-title");
                    if (articleTitle) {
                        openArticleByTitle(articleTitle);
                    }
                });
            });

            // Run on page load
            checkHashOnLoad();

            // Listen for hash changes (back/forward buttons)
            window.addEventListener("hashchange", checkHashOnLoad);
        })();
        </script>';
        
        return $html;
    }

    /**
     * Add admin menu
     */
    public function addAdminMenu()
    {
        add_options_page(
            'Blinkz Configuration',
            'Blinkz Config',
            'manage_options',
            'blinkz-config',
            [$this, 'configurationPage']
        );
    }

    /**
     * Add settings link on plugin page
     */
    public function addSettingsLink($links)
    {
        $settings_link = '<a href="' . admin_url('options-general.php?page=blinkz-config') . '">Settings</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Initialize settings
     */
    public function initSettings()
    {
        register_setting('blinkz_settings', 'blinkz_options');
    }

    /**
     * Configuration page HTML
     */
    public function configurationPage()
    {
        $options = get_option('blinkz_options', []);
        $api_url = Arr::get($options, 'api_url', self::DEFAULT_API_URL);
        $custom_css = Arr::get($options, 'custom_css', self::DEFAULT_CSS);
        $has_custom_css = Arr::has($options, 'custom_css');
        
        ?>
        <div class="wrap">
            <h1>Blinkz Configuration</h1>
            
            <?php if (isset($_GET['saved']) && $_GET['saved'] === '1'): ?>
                <div class="notice notice-success is-dismissible">
                    <p>Configuration saved successfully!</p>
                </div>
            <?php endif; ?>

            <form method="post" action="<?php echo admin_url('admin-post.php'); ?>">
                <?php wp_nonce_field('blinkz_config_save', 'blinkz_nonce'); ?>
                <input type="hidden" name="action" value="blinkz_save_config">
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="blinkz_api_url">API URL Endpoint</label>
                        </th>
                        <td>
                            <input type="url"
                                   id="blinkz_api_url"
                                   name="blinkz_options[api_url]"
                                   value="<?php echo esc_attr($api_url); ?>"
                                   class="regular-text"
                                   placeholder="https://api.example.com/endpoint">
                            <p class="description">Enter the Blinkz API endpoint URL</p>
                            <button type="button" id="restore-default-api-url" class="button">Restore Default API URL</button>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="blinkz_intro_text">Intro Text</label>
                        </th>
                        <td>
                            <textarea id="blinkz_intro_text" 
                                      name="blinkz_options[intro_text]" 
                                      rows="3" 
                                      class="large-text"><?php echo esc_textarea(Arr::get($options, 'intro_text', self::DEFAULT_INTRO_TEXT)); ?></textarea>
                            <p class="description">Text displayed above the category buttons. This will be followed by: "Staat jouw vraag er niet bij, neem dan contact op met het Blinkz Informatiepunt."</p>
                            <button type="button" id="restore-default-intro" class="button">Restore Default Intro Text</button>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="blinkz_show_newest">Show Newest Articles</label>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox"
                                       id="blinkz_show_newest"
                                       name="blinkz_options[show_newest]"
                                       value="1"
                                       <?php checked(Arr::get($options, 'show_newest', self::DEFAULT_SHOW_NEWEST)); ?>>
                                Enable display of newest articles section
                            </label>
                            <p class="description">When enabled, shows a section with the newest articles on the page.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="blinkz_show_most_read">Show Most Read Articles</label>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox"
                                       id="blinkz_show_most_read"
                                       name="blinkz_options[show_most_read]"
                                       value="1"
                                       <?php checked(Arr::get($options, 'show_most_read', self::DEFAULT_SHOW_MOST_READ)); ?>>
                                Enable display of most read articles section
                            </label>
                            <p class="description">When enabled, shows a section with the most read articles on the page.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="blinkz_articles_amount">Articles Amount</label>
                        </th>
                        <td>
                            <input type="number"
                                   id="blinkz_articles_amount"
                                   name="blinkz_options[articles_amount]"
                                   value="<?php echo esc_attr(Arr::get($options, 'articles_amount', self::DEFAULT_ARTICLES_AMOUNT)); ?>"
                                   class="small-text"
                                   min="0"
                                   max="50">
                            <p class="description">Number of articles to show in the newest/most-read sections. Set to 0 to hide these sections.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="blinkz_custom_css">Custom CSS</label>
                        </th>
                        <td>
                            <textarea id="blinkz_custom_css"
                                      name="blinkz_options[custom_css]"
                                      rows="20"
                                      class="large-text code"
                                      style="font-family: monospace;"><?php echo esc_textarea($custom_css); ?></textarea>
                            <p class="description">Customize the appearance of the Blinkz Q&A block</p>
                            <button type="button" id="restore-default-css" class="button">Restore Default CSS</button>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button('Save Configuration'); ?>
            </form>
            
            <hr>
            
            <h2>Current Configuration</h2>
            <table class="widefat">
                <thead>
                    <tr>
                        <th>Setting</th>
                        <th>Value</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td>API URL Endpoint</td>
                        <td><?php
                            $has_api_url = Arr::has($options, 'api_url');

                            if (!$has_api_url) {
                                echo '<em>Using default</em>';
                            } else {
                                $normalized_api_url = trim($api_url);
                                $normalized_default_api_url = trim(self::DEFAULT_API_URL);
                                echo ($normalized_api_url !== $normalized_default_api_url) ? '<em>Customized</em>' : '<em>Using default</em>';
                            }
                        ?></td>
                    </tr>
                    <tr>
                        <td>Intro Text</td>
                        <td><?php
                            $intro_text = Arr::get($options, 'intro_text', self::DEFAULT_INTRO_TEXT);
                            $has_intro_text = Arr::has($options, 'intro_text');

                            // Check if intro text exists in options and if it's different from default
                            if (!$has_intro_text) {
                                echo '<em>Using default</em>';
                            } else {
                                // Normalize both strings for comparison
                                $normalized_intro = trim($intro_text);
                                $normalized_default_intro = trim(self::DEFAULT_INTRO_TEXT);
                                echo ($normalized_intro !== $normalized_default_intro) ? '<em>Customized</em>' : '<em>Using default</em>';
                            }
                        ?></td>
                    </tr>
                    <tr>
                        <td>Show Newest Articles</td>
                        <td><?php echo Arr::get($options, 'show_newest', self::DEFAULT_SHOW_NEWEST) ? 'Enabled' : 'Disabled'; ?></td>
                    </tr>
                    <tr>
                        <td>Show Most Read Articles</td>
                        <td><?php echo Arr::get($options, 'show_most_read', self::DEFAULT_SHOW_MOST_READ) ? 'Enabled' : 'Disabled'; ?></td>
                    </tr>
                    <tr>
                        <td>Articles Amount</td>
                        <td><?php echo esc_html(Arr::get($options, 'articles_amount', self::DEFAULT_ARTICLES_AMOUNT)); ?></td>
                    </tr>
                    <tr>
                        <td>Custom CSS</td>
                        <td><?php
                            // Check if custom CSS exists in options and if it's different from default
                            if (!$has_custom_css) {
                                echo '<em>Using default</em>';
                            } else {
                                // Normalize both CSS strings for comparison
                                $normalized_custom = preg_replace('/\s+/', ' ', trim($custom_css));
                                $normalized_default = preg_replace('/\s+/', ' ', trim(self::DEFAULT_CSS));
                                echo ($normalized_custom !== $normalized_default) ? '<em>Customized</em>' : '<em>Using default</em>';
                            }
                        ?></td>
                    </tr>
                </tbody>
            </table>

            <hr>

            <p style="color: #666; font-size: 0.9em;">
                Plugin developed by <a href="https://blijnder.nl" target="_blank">Blijnder</a>
            </p>
        </div>

        <script>
        jQuery(document).ready(function($) {
            $('#restore-default-css').on('click', function() {
                if (confirm('Are you sure you want to restore the default CSS?')) {
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'blinkz_restore_default_css',
                            _ajax_nonce: '<?php echo wp_create_nonce('blinkz_restore_css'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                $('#blinkz_custom_css').val(response.data.css);
                                alert('Default CSS restored. Remember to save the configuration.');
                            } else {
                                alert('Failed to restore default CSS.');
                            }
                        }
                    });
                }
            });
            
            $('#restore-default-intro').on('click', function() {
                if (confirm('Are you sure you want to restore the default intro text?')) {
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'blinkz_restore_default_intro',
                            _ajax_nonce: '<?php echo wp_create_nonce('blinkz_restore_intro'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                // Decode the text to handle any escaped characters
                                var defaultText = $('<textarea />').html(response.data.intro_text).text();
                                $('#blinkz_intro_text').val(defaultText);
                                alert('Default intro text restored. Remember to save the configuration.');
                            } else {
                                alert('Failed to restore default intro text.');
                            }
                        }
                    });
                }
            });

            $('#restore-default-api-url').on('click', function() {
                if (confirm('Are you sure you want to restore the default API URL?')) {
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'blinkz_restore_default_api_url',
                            _ajax_nonce: '<?php echo wp_create_nonce('blinkz_restore_api_url'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                $('#blinkz_api_url').val(response.data.api_url);
                                alert('Default API URL restored. Remember to save the configuration.');
                            } else {
                                alert('Failed to restore default API URL.');
                            }
                        }
                    });
                }
            });
        });
        </script>
        <?php
    }

    /**
     * Save configuration
     */
    public function saveConfiguration()
    {
        // Verify nonce
        if (!wp_verify_nonce($_POST['blinkz_nonce'], 'blinkz_config_save')) {
            wp_die('Security check failed');
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }

        // Get posted data
        $posted_options = isset($_POST['blinkz_options']) ? $_POST['blinkz_options'] : [];
        
        // Get existing options to preserve data
        $options = get_option('blinkz_options', []);
        
        // Handle API URL
        if (Arr::has($posted_options, 'api_url')) {
            $api_url = esc_url_raw(Arr::get($posted_options, 'api_url'));
            $options['api_url'] = $api_url;
        } else {
            $options['api_url'] = '';
        }
        
        // Handle Intro Text
        if (Arr::has($posted_options, 'intro_text')) {
            $intro_text = wp_kses_post(Arr::get($posted_options, 'intro_text'));
            $options['intro_text'] = $intro_text;
        } else {
            $options['intro_text'] = '';
        }
        
        // Handle Show Newest Articles (checkbox)
        $options['show_newest'] = Arr::has($posted_options, 'show_newest') ? true : false;

        // Handle Show Most Read Articles (checkbox)
        $options['show_most_read'] = Arr::has($posted_options, 'show_most_read') ? true : false;

        // Handle Articles Amount
        $articles_amount = (int) Arr::get($posted_options, 'articles_amount', self::DEFAULT_ARTICLES_AMOUNT);
        $options['articles_amount'] = max(0, min(50, $articles_amount)); // Clamp between 0 and 50

        // Handle Custom CSS
        if (Arr::has($posted_options, 'custom_css')) {
            // Sanitize CSS but preserve the formatting
            $custom_css = wp_strip_all_tags(Arr::get($posted_options, 'custom_css'));

            // Only save custom CSS if it's different from default
            $normalized_custom = preg_replace('/\s+/', ' ', trim($custom_css));
            $normalized_default = preg_replace('/\s+/', ' ', trim(self::DEFAULT_CSS));

            if ($normalized_custom !== $normalized_default) {
                $options['custom_css'] = $custom_css;
            } else {
                // If it matches default, remove it from options
                unset($options['custom_css']);
            }
        }

        // Save options
        update_option('blinkz_options', $options);

        // Redirect back with success message
        wp_redirect(admin_url('options-general.php?page=blinkz-config&saved=1'));
        exit;
    }
    
    /**
     * AJAX handler to restore default CSS
     */
    public function restoreDefaultCss()
    {
        // Verify nonce
        if (!wp_verify_nonce($_POST['_ajax_nonce'], 'blinkz_restore_css')) {
            wp_send_json_error('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        // Return default CSS
        wp_send_json_success([
            'css' => self::DEFAULT_CSS
        ]);
    }
    
    /**
     * AJAX handler to restore default intro text
     */
    public function restoreDefaultIntro()
    {
        // Verify nonce
        if (!wp_verify_nonce($_POST['_ajax_nonce'], 'blinkz_restore_intro')) {
            wp_send_json_error('Security check failed');
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        // Send the response with proper encoding
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode([
            'success' => true,
            'data' => [
                'intro_text' => self::DEFAULT_INTRO_TEXT
            ]
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        wp_die();
    }

    /**
     * AJAX handler to restore default API URL
     */
    public function restoreDefaultApiUrl()
    {
        // Verify nonce
        if (!wp_verify_nonce($_POST['_ajax_nonce'], 'blinkz_restore_api_url')) {
            wp_send_json_error('Security check failed');
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        // Return default API URL
        wp_send_json_success([
            'api_url' => self::DEFAULT_API_URL
        ]);
    }
}

/**
 * Helper class for array operations
 */
class Arr
{
    /**
     * Get value from array by key
     */
    public static function get($array, $key, $default = null)
    {
        return isset($array[$key]) ? $array[$key] : $default;
    }

    /**
     * Check if key exists in array
     */
    public static function has($array, $key)
    {
        return isset($array[$key]);
    }
}

// Initialize plugin
BlinkzPlugin::getInstance();

/**
 * Helper function to get Blinkz option
 */
function blinkz_get_option($key, $default = null)
{
    $options = get_option('blinkz_options', []);
    return Arr::has($options, $key) ? Arr::get($options, $key) : $default;
}

/**
 * Plugin activation hook
 */
register_activation_hook(__FILE__, function() {
    $options = get_option('blinkz_options', []);

    // Set default CSS if not already set
    if (!Arr::has($options, 'custom_css')) {
        $options['custom_css'] = BlinkzPlugin::DEFAULT_CSS;
    }

    // Set default intro text if not already set
    if (!Arr::has($options, 'intro_text')) {
        $options['intro_text'] = BlinkzPlugin::DEFAULT_INTRO_TEXT;
    }

    // Set default API URL if not already set
    if (!Arr::has($options, 'api_url')) {
        $options['api_url'] = BlinkzPlugin::DEFAULT_API_URL;
    }

    // Set default show newest articles if not already set
    if (!Arr::has($options, 'show_newest')) {
        $options['show_newest'] = BlinkzPlugin::DEFAULT_SHOW_NEWEST;
    }

    // Set default show most read articles if not already set
    if (!Arr::has($options, 'show_most_read')) {
        $options['show_most_read'] = BlinkzPlugin::DEFAULT_SHOW_MOST_READ;
    }

    // Set default articles amount if not already set
    if (!Arr::has($options, 'articles_amount')) {
        $options['articles_amount'] = BlinkzPlugin::DEFAULT_ARTICLES_AMOUNT;
    }

    update_option('blinkz_options', $options);
});
